#!/bin/bash

# 检查并安装 OpenSSL
install_openssl() {
    if [[ $release =~ Ubuntu ]]; then
        sudo apt-get update
        sudo apt-get install -y openssl
    elif [[ $release =~ CentOS ]]; then
        sudo yum install -y openssl openssl-devel
    fi
}

# 检测系统版本
detect_system() {
    if [[ -f /etc/os-release ]]; then
        . /etc/os-release
        release="$NAME"
    elif type lsb_release >/dev/null 2>&1; then
        release=$(lsb_release -si)
    else
        echo "无法识别的Linux发行版。"
        exit 1
    fi
}

# 主函数
main() {
    # 检测系统版本
    detect_system

    # 检查并安装 OpenSSL
    install_openssl

    # 获取输入的IP地址
    IP=$1
    if [ -z "$IP" ]; then
        echo "请提供一个IP地址作为参数。"
        exit 1
    fi

    # 设置域名和扩展域名
    DOMAIN=$IP
    DOMAIN_EXT=$IP

    # 设置证书有效期
    DATE=7300

    # 输出IP信息
    echo 'ip为 '$IP

    # 设置证书目录
    CERT_DIR="/data/devops"
    mkdir -p $CERT_DIR

    # 清除旧证书
    rm -rf ${CERT_DIR}/${DOMAIN} ca.key ca.csr ca.crt

    # 创建新目录
    mkdir -p ${CERT_DIR}/${DOMAIN}

    # 生成CA根证书
    ## 准备ca配置文件，得到ca.conf
    cat > ${CERT_DIR}/${DOMAIN}/ca.conf << EOF
[ req ]
default_bits       = 4096
distinguished_name = req_distinguished_name

[ req_distinguished_name ]
countryName                 = Country Name (2 letter code)
countryName_default         = CN
stateOrProvinceName         = State or Province Name (full name)
stateOrProvinceName_default = BeiJing
localityName                = Locality Name (eg, city)
localityName_default        = BeiJing
organizationName            = Organization Name (eg, company)
organizationName_default    = shengzhiyuheng
commonName                  = Common Name (e.g. server FQDN or YOUR name)
commonName_max              = 64
commonName_default          = shengzhiyuheng CA Center
EOF

    ## 生成ca秘钥，得到ca.key
    openssl genrsa -out ${CERT_DIR}/${DOMAIN}/ca.key 4096

    ## 生成ca证书签发请求，得到ca.csr
    openssl req -new -subj "/C=CN/ST=BeiJing/L=BeiJing/O=shengzhiyuheng/CN=shengzhiyuheng CA Center" -sha256 -out ${CERT_DIR}/${DOMAIN}/ca.csr -key ${CERT_DIR}/${DOMAIN}/ca.key -config ${CERT_DIR}/${DOMAIN}/ca.conf

    ## 生成ca根证书，得到ca.crt
    openssl x509 -req -days ${DATE} -in ${CERT_DIR}/${DOMAIN}/ca.csr -signkey ${CERT_DIR}/${DOMAIN}/ca.key -out ${CERT_DIR}/${DOMAIN}/ca.crt

    # 生成终端用户证书
    ## 准备配置文件，得到server.conf
    cat > ${CERT_DIR}/${DOMAIN}/server.conf << EOF
[ req ]
default_bits       = 2048
distinguished_name = req_distinguished_name
req_extensions     = req_ext

[ req_distinguished_name ]
countryName                 = Country Name (2 letter code)
countryName_default         = CN
stateOrProvinceName         = State or Province Name (full name)
stateOrProvinceName_default = BeiJing
localityName                = Locality Name (eg, city)
localityName_default        = BeiJing
organizationName            = Organization Name (eg, company)
organizationName_default    = shengzhiyuheng
commonName                  = zhaorx (e.g. server FQDN or YOUR name)
commonName_max              = 64
EOF
    echo commonName_default          "=" ${DOMAIN} >> ${CERT_DIR}/${DOMAIN}/server.conf
    cat >> ${CERT_DIR}/${DOMAIN}/server.conf << EOF

[ req_ext ]
subjectAltName = @alt_names

[ alt_names ]
EOF
    echo DNS.1 = ${DOMAIN} >> ${CERT_DIR}/${DOMAIN}/server.conf
    echo DNS.2 = ${DOMAIN_EXT} >> ${CERT_DIR}/${DOMAIN}/server.conf
    echo IP    = ${IP} >> ${CERT_DIR}/${DOMAIN}/server.conf

    ## 生成秘钥，得到server.key
    openssl genrsa -out ${CERT_DIR}/${DOMAIN}/server.key 2048

    ## 生成证书签发请求，得到server.csr
    openssl req -new -subj "/C=CN/ST=BeiJing/L=BeiJing/O=shengzhiyuheng/CN=${DOMAIN}" -sha256 -out ${CERT_DIR}/${DOMAIN}/server.csr -key ${CERT_DIR}/${DOMAIN}/server.key -config ${CERT_DIR}/${DOMAIN}/server.conf

    ## 用CA证书生成终端用户证书，得到server.crt
    openssl x509 -req -days 7300 -CA ${CERT_DIR}/${DOMAIN}/ca.crt -CAkey ${CERT_DIR}/${DOMAIN}/ca.key -CAcreateserial -in ${CERT_DIR}/${DOMAIN}/server.csr -out ${CERT_DIR}/${DOMAIN}/server.crt -extensions req_ext -extfile ${CERT_DIR}/${DOMAIN}/server.conf

    # 打印证书有效期
    echo "私有证书有效期：20年"

    # 打印证书目录地址
    echo "证书目录地址：${CERT_DIR}/${DOMAIN}"
}

# 调用主函数并传入参数
main "$@"
